#!/bin/sh
OVERLAYSIZE="$((16*1024*1024))"

fatal() {
    echo "$@" 1>&2
    exit 1
}
say() {
    echo "$@"
}

find_ubi_chardev() {
    local volname="$1"
    local ubiname="$2"
    local ubi=$(grep -E "^$volname\$" /sys/class/ubi/${ubiname}_*/name | cut -f5 -d/)
    [ -n "$ubi" ] && ubi="/dev/$ubi"
    echo "$ubi"
}

find_ubi_volid() {
    local ubi="$(find_ubi_chardev "$1" "$2")"
    echo "${ubi##*_}"
}

ACTIVE=$(/sbin/bootswitcher get-active)
[ "$ACTIVE" -eq 0 -o "$ACTIVE" -eq 1 ] || fatal "Can't determine active set"
OVERLAYNAME="k$(($ACTIVE ^ 1))_data"

# Remove any existing overlay.
id="$(find_ubi_volid "$OVERLAYNAME" "ubi0")"
if [ -n "$id" ]; then
    say  "Removing volume $OVERLAYNAME at UBI index $id..."
    ubirmvol "/dev/ubi0" -n "$id" || fatal "ubirmvol failed"
fi

# Make the new overlay.
say "Creating overlay $OVERLAYNAME of size $OVERLAYSIZE..."
ubimkvol "/dev/ubi0" -N "$OVERLAYNAME" -s "$OVERLAYSIZE" -t dynamic || fatal "ubimkvol failed"

# Mount the new overlay.
say "Mounting the new volume..."
OVLAYMT="/tmp/newoverlay"
mkdir -p "$OVLAYMT" && \
    mount -t ubifs -o noatime "$(find_ubi_chardev "$OVERLAYNAME" ubi0)" "$OVLAYMT" || \
    fatal "Failed to mount new volume"

# Preserve a file (copy from current filesystem to new overlay).
preserve() {
    local srcpath="$1"
    # Ignore if the file doesn't exist.
    [ -e "$srcpath" ] || return 0
    # Ignore if the file is unchanged from the default version.
    cmp -s "$srcpath" "/rom$srcpath" 2> /dev/null && return 0
    # Preserve the file.
    local dst="$OVLAYMT$srcpath"
    say "Preserving $srcpath..."
    mkdir -p "$(dirname "$dst")" && cp -p "$srcpath" "$dst"
}

# Preserve files from current overlay.
say "Copying files from current overlay..."
for f in \
    /home/commapp_state.txt \
    /etc/wpa_supplicant.conf \
    /etc/ssh/ssh_host_*_key* \
    /home/data_logger/config.lua home/data_logger/config.bak \
    /home/data_logger/devices.lua home/data_logger/devices.bak \
    /home/data_logger/meta.lua \
    /home/data_logger/ctrlmsgq.bin \
    /home/data_logger/mime.cfg \
    /home/data_logger/sma_model/* \
    /home/data_logger/sunspec/* \
    /home/data_logger/modbus_map.lua \
    /home/communicator/communicator.cfg \
    /home/mime/mime.cfg \
    /etc/config/pvs5 \
    /etc/firewall.user \
    /etc/dhcp.leases.bak
do
    preserve "$f" || { umount "$OVLAYMT"; fatal "Failed to copy $f"; }
done

NEWDLCFG="$OVLAYMT/home/data_logger/config.lua"

# Add sunspec_ip_range section to data logger config file.
grep -q 'Automatically added SUNSPEC_IP_RANGE' "$NEWDLCFG" || \
    echo '
-- Automatically added SUNSPEC_IP_RANGE
if param_table["sunspec_ip_range"] == nil then
  param_table["sunspec_ip_range"] = [[172.27.153.50;172.27.153.99]]
end
' >> "$NEWDLCFG" || fatal "Failed to add sunspec ip range to scan"

# Add zero_export section to data logger config file.
grep -q 'Automatically added ZERO_EXPORT_V1' "$NEWDLCFG" || \
    echo '
-- Automatically added ZERO_EXPORT_V1
if param_table["zero_export"] == nil then
  param_table["zero_export"] = {}
end
' >> "$NEWDLCFG" || fatal "Failed to add zero-export section"

# Reduce control-channel connection frequency to once per 30 mins.
grep -q 'Automatically added CHEWBACCA_CMD_CHAN_FREQ' "$NEWDLCFG" || \
    echo '
-- Automatically added CHEWBACCA_CMD_CHAN_FREQ
if param_table["server_comms_flags"] == nil then
  param_table["cmd_chan_conn_int"] = 1800
  param_table["cmd_chan_fail_int"] = 600
end
' >> "$NEWDLCFG" || fatal "Failed to change cmd-chan poll frequency"

# Add line to firewall to suppress costly SMA traffic.
if [ -e "$OVLAYMT/etc/firewall.user" ] && \
    ! grep -q "dport 9523" "$OVLAYMT/etc/firewall.user"; then
    echo '
# Block costly SMA traffic to port 9523.
iptables -I FORWARD -p udp --dport 9523 -j DROP
' >> "$OVLAYMT/etc/firewall.user" || fatal "Failed to add SMA firewall rule"
fi

# patch wpa_supplicant.conf
if [ -f /etc/wpa_supplicant.conf ]; then
    sed -i "/disabled=1/d" "$OVLAYMT/etc/wpa_supplicant.conf"
    sed -i '/^network/ a disabled=1' "$OVLAYMT/etc/wpa_supplicant.conf"
    sed -i "/bgscan=.*/d" "$OVLAYMT/etc/wpa_supplicant.conf"
    sed -i '/^network/ a bgscan="simple:30:-45:300"' "$OVLAYMT/etc/wpa_supplicant.conf"
fi

# route mime's splunk messages through SFA
sed -i "s/splunkMode.*/splunkMode\"\: 4,/" "$OVLAYMT/home/mime/mime.cfg"

umount "$OVLAYMT" || fatal "Failed to unmount overlay"

exit 0
