var fs = require('fs');
var path = require('path');
var util = require('util');
var constants_1 = require('./util/constants');
var cmdLoader_1 = require('./cmdLoader');
var argData_1 = require('./util/argData');
var buildUsage_1 = require('./util/buildUsage');
var environments_1 = require("./util/environments");
var pjson = require('../package.json');
var _ = require('underscore');
var callerId = require('caller-id');
var Promise = require('bluebird');
var api = require('./api');
var promptData_1 = require('./util/promptData');
var ExtendedCommand = require('./extendedcommand');
var log = require('./util/logging');
var utilsCore = require('./util/utilsCore');
var S = utilsCore.getLocaleString;
var nodeJSSuperProperty = 'super_';
var DownloadFile = require('./util/downloadFile');
function CtmCli(name, parent, genMode) {
    this.parent = parent;
    this.output = log;
    CtmCli[nodeJSSuperProperty].call(this, name);
    var env = environments_1.environments.getCurrentEnvironment();
    env = env ? env : new environments_1.IEnvironment("", "", "");
    this.targetUrl = env.endPoint;
    this.currentEnv = env;
    this.api = api;
    this.fileDataContent = null;
    this.userSearchOptions = {};
    this.interactiveOption = null;
    this.otputfileOption = null;
    this.additionalYamlProps = null;
    this.isCommandHidden = false;
    this.annotation = {};
    this.utilsCore = utilsCore;
    this.PromptData = promptData_1.PromptData;
    var self = this;
    this.successHandler = function (response, newPassword) {
        var _this = this;
        var p = new Promise(function (resolve, reject) {
            self.info(response.body);
            if (newPassword) {
                _this.updateEnvironment(newPassword);
            }
            resolve("");
        }).then(function (msg) {
            var nextOp = new Promise(function (resolve, reject) {
                resolve("");
            });
            if (self.interactiveOption) {
                if (!self.additionalYamlProps) {
                    throw util.format("To support -i for command:'%s' x-additional-prop: {'interactive': targetUrlProperty } must be defined ", self.fullName());
                }
                var targetUrlProperty = self.additionalYamlProps["interactive"];
                if (!targetUrlProperty) {
                    throw util.format("To support -i for command:'%s' x-additional-prop: {'interactive': targetUrlProperty } must be defined ", self.fullName());
                }
                var targetUrl = response.body[targetUrlProperty];
                if (!targetUrl) {
                    throw util.format("-i is not supported. The interactive URL property '%s' not found in command result", targetUrlProperty);
                }
                nextOp = utilsCore.launchBrowserPromis(targetUrl);
            }
            if (self.outputfileOption) {
                if (!self.additionalYamlProps) {
                    throw util.format("To support -o for command:'%s' x-additional-prop: {'outputfile': targetUrlProperty } must be defined ", self.fullName());
                }
                var outputfileUrlProperty = self.additionalYamlProps["outputfile"];
                if (!outputfileUrlProperty) {
                    throw util.format("To support -o for command:'%s' x-additional-prop: {'outputfile': targetUrlProperty } must be defined ", self.fullName());
                }
                var downloadtUrl = response.body[outputfileUrlProperty];
                if (!downloadtUrl) {
                    throw util.format("-o is not supported. The outputfile URL property '%s' not found in command result", outputfileUrlProperty);
                }
                self.info(util.format("Downloading '%s' into '%s'", downloadtUrl, self.outputfileOption));
                var dl = new DownloadFile(downloadtUrl, self.outputfileOption);
                nextOp = dl.download();
            }
            return nextOp;
        }).then(function (Msg) {
            process.exit(0);
        }).catch(function (error) {
            self.error(error);
            process.exit(1);
        });
    };
    this.errorHandler = function (error) {
        if (error && error.response) {
            var response = error.response;
            var jsonBody = self.stringify(response.body);
            if (self.json()) {
                console.error(jsonBody);
            }
            else {
                self.output.error(S('Failed: status code %s\n%s', response.statusCode, jsonBody));
            }
        }
        else {
            if (error.message && error.message.indexOf('form-data: ENOENT') >= 0 && error.message.indexOf('form-data: ENOENT: no such file') < 0) {
                error.message = error.message.replace('form-data: ENOENT', 'form-data: ENOENT: no such file or directory');
            }
            self.error(error);
        }
        process.exit(1);
    };
    this.updateEnvironment = function (password) {
        var currentEnvironment = environments_1.environments.getCurrentEnvironment();
        var currentEnvironmentName = environments_1.environments.findEnvironmentName(currentEnvironment);
        environments_1.environments.updateEnvironment(currentEnvironmentName, "password", password);
    };
    this.setApiKey = function (module, token) {
        module.apiKey = 'Bearer ' + token;
    };
    this.getAuthenticatedModule = function (moduleConstructor) {
        var _this = this;
        var env = this.currentEnv;
        if (!env.endPoint) {
            var envName = environments_1.environments.findEnvironmentName(env);
            if (!envName) {
                return Promise.reject(new Error(S('No environment is set. If you want to connect to a Control-M instance,' +
                    ' run the following command: \'ctm environment add controlm https://<webhost>:8443/automation-api <user> <password>\'. ' +
                    'If you want to connect to a workbench, run the following command: \'ctm environment workbench::add\'. ' +
                    'For information on how to configure an environment, run \'ctm environment\'.')));
            }
            else {
                return Promise.reject(new Error(S('endPoint was not specified for environment \'%s\'. ' +
                    'Use \'ctm environment update %s endPoint <endPoint>\' to specify an endPoint for the environment.', envName, envName)));
            }
        }
        var module = new moduleConstructor('', env.endPoint);
        if (moduleConstructor === self.api.SessionApi) {
            return Promise.resolve(module);
        }
        if (this.userToken) {
            this.setApiKey(module, this.userToken);
            return Promise.resolve(module);
        }
        return new self.api.SessionApi('', env.endPoint, undefined, undefined, env).doLogin({}, new self.api.LoginCredentials(env.user, env.decriptedPassword)).then(function (result) {
            _this.setApiKey(module, result.body.token);
            if (!result.body.version) {
                _this.errorHandler(new Error("You are running ctm-cli of version " + pjson.version + " against server of lower version. Please download and install matching client version from " + env.endPoint + "/ctm-cli.tgz"));
            }
            if (result.body.version != pjson.version) {
                _this.errorHandler(new Error("You are running ctm-cli of version " + pjson.version + " against server of version " + result.body.version + ". Please download and install matching client version from " + env.endPoint + "/ctm-cli.tgz"));
            }
            return module;
        }).disposer(function () {
            var logout = new self.api.SessionApi('', env.endPoint);
            logout.apiKey = module.authentications.Bearer.apiKey;
            return logout.doLogout();
        });
    };
    Object.defineProperty(Error.prototype, 'message', {
        configurable: true,
        enumerable: true
    });
    if (parent) {
        this._mode = parent._mode;
    }
    else {
        this.initSetup();
        this.enableNestedCommands(this);
        this.checkVersion();
        this._mode = genMode || constants_1.Constants.API_VERSIONS.CTM;
        var loader = new cmdLoader_1.CmdLoader(this, this._mode);
        if (genMode) {
            log.info('Generating command metadata file: ' + loader.cmdMetadataFile);
            loader.harvestPlugins();
            loader.harvestModules();
            loader.saveCmdMetadata();
            log.info('Done');
            return;
        }
        else if (loader.cmdMetadataExists()) {
            loader.initFromCmdMetadata(CtmCli);
        }
        else {
            log.warn('No existing command metadata files. Command will run slow.');
            loader.harvestPlugins();
            loader.harvestModules();
        }
    }
}
util.inherits(CtmCli, ExtendedCommand);
_.extend(CtmCli.prototype, {
    initSetup: function () {
        var self = this;
        self.debug = process.env.AZURE_DEBUG === '1';
        if (!self.debug && process.listeners('uncaughtException').length === 0) {
            self.uncaughExceptionHandler = function (err) {
                var noConsole = false;
                if (!log['default'].transports.console) {
                    noConsole = true;
                    self.output.add(self.output.transports.Console);
                }
                var loggedFullError = false;
                if (err.message) {
                    log.error(err.message);
                }
                else if (err.Message) {
                    log.error(err.Message);
                }
                else {
                    log.json('error', err);
                    loggedFullError = true;
                }
                if (!loggedFullError) {
                    if (err.stack) {
                        log.verbose('stack', err.stack);
                    }
                    log.json('silly', err);
                }
                self.recordError(err);
                if (noConsole) {
                    self.output.remove(self.output.transports.Console);
                }
                self.exit(1, 'error', null);
            };
            process.addListener('uncaughtException', self.uncaughExceptionHandler);
        }
    },
    getErrorFile: function () {
        return path.join(utilsCore.ctmDir(), 'ctm.err');
    },
    getSillyErrorFile: function () {
        return path.join(utilsCore.ctmDir(), 'ctm.details.err');
    },
    recordError: function (err) {
        if (err) {
            var errorFile = this.getErrorFile();
            try {
                var writeFileFunction = fs.appendFileSync;
                writeFileFunction(errorFile, (new Date()) + ':\n' +
                    util.inspect(err) + '\n' + err.stack + '\n');
                if (!this.json()) {
                    log.error('Error information has been recorded to ' + errorFile);
                }
            }
            catch (err2) {
                log.warn('Cannot save error information :' + util.inspect(err2));
            }
            log.writeCapturedSillyLogs(this.getSillyErrorFile(), process.env.AZURE_CLI_APPEND_LOGS);
        }
    },
    exit: function (exitCode, level, message) {
        var self = this;
        if (message) {
            log.log(level, message);
        }
        if (self.uncaughtExceptionHandler) {
            process.removeListener('uncaughtException', self.uncaughExceptionHandler);
        }
        process.exit(exitCode);
    },
    normalizeAuthorizationError: function (msg) {
        var regex = /.*The \'Authorization\' header is not present or provided in an invalid format.*/ig;
        if (msg.match(regex)) {
            msg = 'Certificate based Authentication is not supported in current mode: \'' + this._mode +
                '\'. Please authenticate using an organizational account via \'ctm login\' command.';
        }
        return msg;
    },
    execute: function (fn) {
        var self = this;
        return self.action(function () {
            self.setupCommandOutput();
            if (log.format().json) {
                log.verbose('Executing command ' + self.fullName().bold);
            }
            else {
                log.info('Executing command ' + self.fullName().bold);
            }
            try {
                var argsCount = fn.length <= 1 && self._args ? self._args.length + 2 : fn.length;
                var args = new Array(argsCount);
                var optionIndex = arguments.length - 1;
                for (var i = 0; i < arguments.length; i++) {
                    if (typeof arguments[i] === 'object') {
                        optionIndex = i;
                        break;
                    }
                }
                var options = arguments[optionIndex].optionValues;
                args[args.length - 2] = options;
                args[args.length - 1] = callback;
                var freeArguments = 0;
                for (var j = 0; self._args && j < self._args.length; j++) {
                    var optionName = utilsCore.camelcase(self._args[j].name);
                    if (options && options[optionName]) {
                        args[j] = options[optionName];
                        delete options[optionName];
                    }
                    else if (freeArguments < arguments.length) {
                        args[j] = arguments[freeArguments];
                        freeArguments++;
                    }
                }
                fn.apply(this, args);
            }
            catch (err) {
                callback(err);
            }
            function callback(err) {
                if (err) {
                    var noConsole = false;
                    if (!process.env.AZURE_NO_ERROR_ON_CONSOLE && !log['default'].transports.console) {
                        noConsole = true;
                        self.output.add(self.output.transports.Console);
                    }
                    if (err.message) {
                        self.error(err);
                    }
                    else if (err.Message) {
                        if (typeof err.Message === 'object' && typeof err.Message['#'] === 'string') {
                            var innerError;
                            try {
                                innerError = JSON.parse(err.Message['#']);
                            }
                            catch (e) {
                            }
                            if (innerError) {
                                if (noConsole) {
                                    self.output.remove(self.output.transports.Console);
                                }
                                return callback(innerError);
                            }
                        }
                        err.message = self.normalizeAuthorizationError(err.message);
                        self.error(err.Message);
                        log.json('verbose', err);
                    }
                    else {
                        self.error(err);
                    }
                    self.recordError(err);
                    if (err.stack) {
                        (self.debug ? log.error : log.verbose)(err.stack);
                    }
                    if (noConsole) {
                        self.output.remove(self.output.transports.Console);
                    }
                    if (!self.json()) {
                        log.error(self.fullName().bold + ' command ' + 'failed\n'.red.bold);
                    }
                    self.exit(1);
                }
                else {
                    if (log.format().json) {
                        self.exit(0, 'verbose', self.fullName().bold + ' command ' + 'OK'.green.bold);
                    }
                    else {
                        self.exit(0, 'info', self.fullName().bold + ' command ' + 'OK'.green.bold);
                    }
                }
            }
        });
    },
    parseOptions: function (argv) {
        var args = [];
        var len = argv.length;
        var literal = false;
        var option;
        var arg;
        var unknownOptions = [];
        for (var i = 0; i < len; ++i) {
            arg = argv[i];
            if ('--' === arg) {
                literal = true;
                continue;
            }
            if (literal) {
                args.push(arg);
                continue;
            }
            option = this.optionFor(arg);
            var commandOption = null;
            if (!option && arg[0] === '-') {
                var command = this;
                var arga = null;
                for (var a = 0; a < args.length && command && !commandOption; ++a) {
                    arga = args[a];
                    if (command.categories && (arga in command.categories)) {
                        command = command.categories[arga];
                        commandOption = command.optionFor(arg);
                        continue;
                    }
                    break;
                }
                if (!commandOption && arga && command && command.commands) {
                    for (var j in command.commands) {
                        if (command.commands[j].name === arga) {
                            commandOption = command.commands[j].optionFor(arg);
                            break;
                        }
                    }
                }
            }
            if (option) {
                if (option.required) {
                    arg = argv[++i];
                    if (!arg) {
                        return this.optionMissingArgument(option);
                    }
                    if ('-' === arg[0]) {
                        return this.optionMissingArgument(option, arg);
                    }
                    this.emit(option.name(), arg);
                }
                else if (option.optional) {
                    arg = argv[i + 1];
                    if (!arg || '-' === arg[0]) {
                        arg = null;
                    }
                    else {
                        ++i;
                    }
                    this.emit(option.name(), arg);
                }
                else {
                    this.emit(option.name());
                }
                continue;
            }
            if (arg.length > 1 && '-' === arg[0]) {
                unknownOptions.push(arg);
                commandOption = commandOption || { optional: 1 };
                if (commandOption.required || (commandOption.optional && argv[i + 1] && '-' !== argv[i + 1][0])) {
                    unknownOptions.push(argv[++i]);
                }
                continue;
            }
            args.push(arg);
        }
        return { args: args, unknown: unknownOptions };
    },
    setupCommandLogFormat: function (topMost) {
        if (topMost) {
            var opts = {
                json: false,
                level: 'info',
                logo: 'on'
            };
            log.format(opts);
        }
    },
    setupOptions: function (options) {
        var _this = this;
        var env = environments_1.environments.getCurrentEnvironment();
        env = env ? env : new environments_1.IEnvironment("", "", "");
        if (options && options.environment) {
            var envName = options.environment;
            env = environments_1.environments.getEnvironment(envName);
            if (!env) {
                throw new Error(S('Environment \'%s\' not defined', envName));
            }
        }
        if (options && options.token) {
            this.userToken = options.token;
        }
        if (options && options.file) {
            try {
                var inputFile = options.file;
                if (!path.isAbsolute(inputFile)) {
                    inputFile = path.normalize(path.join(process.cwd(), inputFile));
                }
                this.fileDataContent = require(inputFile);
                Object.keys(this.fileDataContent).forEach(function (e) {
                    if (e[0].toLocaleLowerCase() !== e[0]) {
                        var correctNameProperty = e[0].toLocaleLowerCase() + e.substring(1);
                        _this.fileDataContent[correctNameProperty] = _this.fileDataContent[e];
                        delete _this.fileDataContent[e];
                    }
                });
            }
            catch (error) {
                throw new Error(S('Error using data file \'%s\'. %s', options.file, error));
            }
        }
        if (options && options.search) {
            var sOptions = options.search.split('&');
            var tmpMap = {};
            sOptions.forEach(function (opt) {
                var optData = opt.split('=');
                if (optData.length == 2) {
                    tmpMap[optData[0]] = optData[1];
                }
                else if (optData.length = 1) {
                    tmpMap[optData[0]] = 'true';
                }
                else {
                    throw new Error(S('%s is not a in the expected form of key=value[&key=value]', optData));
                }
            });
            this.userSearchOptions = tmpMap;
        }
        if (options && options.interactive) {
            this.interactiveOption = options.interactive;
        }
        if (options && options.outputfile) {
            try {
                var outputFile = options.outputfile;
                if (!path.isAbsolute(outputFile)) {
                    outputFile = path.normalize(path.join(process.cwd(), outputFile));
                }
                this.outputfileOption = outputFile;
            }
            catch (error) {
                throw new Error(S('Error using -o to file \'%s\'. %s', options.outputfile, error));
            }
        }
        if (options && options.annotation) {
            var errorMsg = "Specified annotation " + options.annotation + " is not in the expected form, must be subject=value&description=value with only 2 fields";
            var values = options.annotation.split('&');
            if (values.length != 2) {
                throw new Error(errorMsg);
            }
            var annotation = {};
            values.forEach(function (value) {
                var data = value.split('=');
                if (data.length == 2 && data[1].length > 0) {
                    annotation[data[0]] = data[1];
                }
                else {
                    throw new Error(errorMsg);
                }
            });
            if (annotation["subject"] == null || annotation["description"] == null) {
                throw new Error(errorMsg);
            }
            if (annotation["subject"].length > 255 || annotation["description"].length > 255) {
                throw new Error("Annotation parsing failed. Subject and description fields cannot be longer than 255 symbols");
            }
            this.annotation = {
                "Annotation-Subject": annotation["subject"],
                "Annotation-Description": annotation["description"]
            };
        }
        this.currentEnv = env;
        this.targetUrl = env.endPoint;
    },
    addOperation: function (className, classVarName, operationId, path, description, actionName, httpMethod, hiddenCommand, args, additionalProperties, additionalDesc, additionalUsage, additionalPromptUsage, searchRequired, futureEnable, internalApi, queryArgs) {
        if (futureEnable === 'true' || internalApi === 'true') {
            hiddenCommand = true;
        }
        if (classVarName === 'session') {
            return;
        }
        var funcArgs = args.slice();
        var bodyArg = null;
        var bodyArgModelArgs = null;
        for (var i = 0; i < args.length; ++i) {
            if (args[i].isBody) {
                bodyArgModelArgs = this.modelsArgs[args[i].type];
                if (!bodyArgModelArgs &&
                    args[i].type !== 'integer' &&
                    args[i].type !== 'number' &&
                    args[i].type !== 'string' &&
                    args[i].type !== 'boolean' &&
                    args[i].type !== 'object' &&
                    args[i].type !== 'array') {
                    this.output.info('Cannot find model for ' + args[i].type);
                    break;
                }
                if (!bodyArgModelArgs) {
                    bodyArgModelArgs = [args[i]];
                }
                bodyArg = args[i];
                funcArgs.splice.apply(funcArgs, [i, 1].concat(bodyArgModelArgs.map(function (arg) { return new argData_1.ArgData(arg.name, arg.type, arg.required, arg.description, arg.hidden, arg.prompt); })));
                break;
            }
        }
        var commandName = operationId;
        var pathParts = path.split('/');
        if (pathParts.length === 2 && funcArgs.length > 0) {
            commandName = constants_1.Constants.EMPTY_COMMAND_NAME;
        }
        else if (pathParts.length >= 3 && pathParts[2][0] !== '{') {
            commandName = pathParts[2];
        }
        var usage = funcArgs.map(function (arg) { return buildUsage_1.buildUsage(arg, operationId); }).join(' ') || ' ';
        var executeArgs = funcArgs.map(function (arg) { return arg.name; }).concat('options, _').join(', ');
        var executeFuncBody = 'try { cli.setupOptions(options);';
        var code = this.additionalPropertiesCodeGen(additionalProperties, path);
        executeFuncBody += code;
        if (bodyArg) {
            bodyArgModelArgs.forEach(function (arg) {
                var name = arg.name;
                executeFuncBody += 'if(cli.fileDataContent && cli.fileDataContent.' + name + '!=null){';
                executeFuncBody += '    if (' + name + '==null){';
                executeFuncBody += '        ' + name + ' = cli.fileDataContent.' + name + ';';
                executeFuncBody += '    }';
                executeFuncBody += '    cli.fileDataContent.' + name + '=null;';
                executeFuncBody += '}';
            });
            if (this.prompt && this.checkIfPromptExists(funcArgs)) {
                executeFuncBody += this.AddPromptToBody(funcArgs);
            }
            executeFuncBody += ' var ' + bodyArg.name + ' = new cli.api.' + bodyArg.type + '(' + bodyArgModelArgs.map(function (arg) { return arg.name; }).join(',') + ');';
            executeFuncBody += ' cli.checkMistakesInInputFile();';
        }
        for (var i = 0; i < args.length; ++i) {
            var arg1 = args[i];
            if (queryArgs.indexOf(arg1.name) > -1) {
                executeFuncBody += ' if (cli.userSearchOptions[\'' + arg1.name + '\']){';
                executeFuncBody += '    if(' + arg1.name + '==null){';
                executeFuncBody += '        ' + arg1.name + ' = cli.userSearchOptions[\'' + arg1.name + '\'];';
                executeFuncBody += '    }';
                executeFuncBody += '    cli.userSearchOptions[\'' + arg1.name + '\']=null;';
                executeFuncBody += ' }';
            }
        }
        executeFuncBody += ' if (Object.keys(cli.userSearchOptions).length > 0 || ' + searchRequired + ') cli.checkMistakesInSearchParameters(' + searchRequired + '\, \'' + queryArgs.toString() + '\');';
        executeFuncBody +=
            'cli.using(cli.getAuthenticatedModule(cli.api.' + className + '), function(module) { ' +
                '    return module.' + operationId + '(cli.annotation' + (args.length > 0 ? "," : "") + args.map(function (arg) {
                return arg.name;
            }).join(',') + ');';
        executeFuncBody += ' }).then(cli.successHandler).catch(cli.errorHandler);';
        if (this.prompt && this.checkIfPromptExists(funcArgs)) {
            executeFuncBody += " }).catch(cli.errorHandler);";
        }
        executeFuncBody += '} catch(exception) {' +
            '   cli.errorHandler(exception);' +
            '}';
        var cli = this;
        cli.using = Promise.using;
        var executeFunc = new Function(executeArgs, executeFuncBody);
        commandName = this.changeCommandStructureIfNeeded(classVarName, description, httpMethod, actionName, path, commandName);
        var command = usage ? commandName + ' ' + usage : commandName;
        description += ' ' + additionalDesc;
        description = description.replace(/&lt;/g, '<').replace(/&gt;/g, '>').replace(/\s+/g, ' ');
        usage += ' ' + additionalUsage + ' ' + additionalPromptUsage;
        usage = usage.replace(/&lt;/g, '<').replace(/&gt;/g, '>').replace(/\s+/g, ' ');
        var newCtmCliCmd = this.addCommand(classVarName, command);
        newCtmCliCmd.isCommandHidden = hiddenCommand;
        newCtmCliCmd.description(description)
            .option('-e, --environment <env>', 'Use <env> environment for this command')
            .option('-t, --token <token>', 'Use the user token <token> for this command')
            .option('-f, --file <input file.json>', 'Use input file for more detailed advanced configurations')
            .option('-s, --search <search query>', 'Use a query to refine/limit the results')
            .option('-i, --interactive', 'View result in an interactive user interface')
            .option('-o, --outputfile <file path>', 'Download results to file')
            .option('-a, --annotation <annotation params>', 'Provide annotation fields for specified operation. Required fields: subject, description ')
            .option('-p, --prompt ', 'Use prompt for this command')
            .execute(executeFunc).usage(usage);
    },
    checkIfPromptExists: function (args) {
        for (var index = 0; index < args.length; index++) {
            var element = args[index];
            if (element && element.prompt) {
                return true;
            }
        }
        return false;
    },
    BuildPromptArray: function (args) {
        var promptDataArray = new Array();
        for (var index = 0; index < args.length; index++) {
            var element = args[index];
            if (element && element.prompt) {
                var validate = element.prompt.indexOf("VALIDATE") !== -1;
                var label = element.prompt;
                if (validate) {
                    label = label.split("VALIDATE")[0].trim();
                }
                var promptData = new promptData_1.PromptData(label, validate, element.name);
                promptDataArray.push(promptData);
            }
        }
        return promptDataArray;
    },
    AddPromptToBody: function (args) {
        return " var promptDataArray = cli.BuildPromptArray(" + JSON.stringify(args) + "); " +
            " cli.processPromisesArray(promptDataArray, cli.utilsCore.promptWithValidate).then(function(values) {" +
            " for (var index = 0; index < promptDataArray.length; index++) { " +
            " eval(promptDataArray[index].elementName + ' = \"' + values[index] + '\"');  } ";
    },
    processPromisesArray: function (valuesArray, func) {
        var index = 0;
        var valuesAnswer = new Array();
        return new Promise(function (resolve, reject) {
            function next() {
                if (index < valuesArray.length) {
                    func(valuesArray[index++]).then(function (value) {
                        valuesAnswer.push(value);
                        next();
                    }, reject);
                }
                else {
                    resolve(valuesAnswer);
                }
            }
            next();
        });
    },
    debugPrint: function (msg) {
        log.info(msg);
    },
    checkMistakesInSearchParameters: function (isRequired, validValues) {
        if (!validValues)
            throw new Error('{errors: [{ message: this cli doesn\'t support -s option }]}');
        var keys = Object.keys(this.userSearchOptions);
        if (isRequired && keys.length == 0) {
            throw new Error('Missing required option `-s, --search`');
        }
        for (var i = 0; i < keys.length; ++i) {
            if (this.userSearchOptions[keys[i]])
                throw new Error('{errors: [{ message: ' + S('%s is not a valid search option, valid values are: %s', keys[i], validValues) + '}]}');
        }
    },
    checkMistakesInInputFile: function () {
        for (var property in this.fileDataContent) {
            if (this.fileDataContent != null && this.fileDataContent[property]) {
                throw new Error('{errors: [{ message: ' + S('invalid property %s in input file', property) + '}]}');
            }
        }
    },
    capitalize: function (word) {
        return word.charAt(0).toUpperCase() + word.slice(1);
    },
    additionalPropertiesCodeGen: function (additionalProperties, path) {
        if (!additionalProperties) {
            return "";
        }
        var additionalPropertiesCodeSnippet = additionalProperties.replace(/&quot;/g, "\"");
        try {
            var tst = JSON.parse(additionalPropertiesCodeSnippet);
        }
        catch (error) {
            throw util.format("faild to eval the additional properties '%s' for command '%s' %s", additionalPropertiesCodeSnippet, path, error);
        }
        return util.format('cli.additionalYamlProps = %s;', additionalPropertiesCodeSnippet);
    },
    changeCommandStructureIfNeeded: function (serviceName, description, httpMethod, actionName, path, commandName) {
        if (serviceName === "samples") {
            return commandName;
        }
        if (commandName === constants_1.Constants.EMPTY_COMMAND_NAME) {
            return commandName;
        }
        if (actionName === "" && httpMethod == "GET")
            actionName = "get";
        if (actionName === "" && httpMethod == "POST")
            actionName = "add";
        if (actionName === "" && httpMethod == "DELETE")
            actionName = "delete";
        path = path.replace(/\/{[^}]*}/g, '');
        path = path.replace(new RegExp(util.format('^/%s', serviceName), "g"), '');
        path = path.replace(/\//g, ':');
        path = path.replace(/^:/g, '');
        commandName = path;
        if (actionName === "!") {
            return commandName;
        }
        commandName = commandName.replace(new RegExp(util.format(':%s$', actionName), "g"), '');
        commandName = commandName + '::' + actionName;
        return commandName;
    },
    info: function (data) {
        data = this.stringify(data);
        if (this.json()) {
            console.log(data);
        }
        else {
            this.output.info(data);
        }
    },
    error: function (error) {
        if (this.json()) {
            console.log(this.stringifyError(error));
        }
        else {
            this.output.error(this.stringify(error.message ? error.message : error));
        }
    },
    wrapError: function (jsonError) {
        return !jsonError || jsonError.errors ? jsonError : { errors: [jsonError] };
    },
    stringify: function (data) {
        return typeof data === 'string' ? data : JSON.stringify(data, undefined, 2);
    },
    stringifyError: function (error) {
        if (typeof error === 'string') {
            return error;
        }
        else if (typeof error === 'object' && error.message) {
            return JSON.stringify(this.wrapError({ message: error.message }), undefined, 2);
        }
        else {
            return JSON.stringify(this.wrapError(error), undefined, 2);
        }
    },
    json: function () {
        return log['default'].transports.console.json;
    },
    setupCommandOutput: function (raw) {
        var self = this;
        var verbose = 0;
        var json = true;
        if (!raw) {
            raw = self.normalize(self.parent.rawArgs.slice(2));
        }
        if (raw.indexOf('help') >= 0) {
            json = false;
        }
        function hasOption(optionName) {
            return self.options.some(function (o) {
                return o.long === optionName;
            });
        }
        for (var i = 0, len = raw.length; i < len; ++i) {
            if (raw[i] === '--json') {
                json = true;
            }
            if (raw[i] === '-v' || raw[i] === '--verbose') {
                verbose = 1;
            }
            else if (raw[i] === '-vv') {
                verbose = 2;
            }
        }
        var opts = {};
        if (verbose || json) {
            if (json) {
                opts.json = true;
                opts.level = 'data';
            }
            if (verbose) {
                opts.json = false;
                if (verbose >= 2) {
                    opts.level = 'silly';
                }
                else {
                    opts.level = 'verbose';
                }
            }
        }
        else {
            opts.level = 'info';
        }
        log.format(opts);
    },
    enableNestedCommands: function (command) {
        if (!command.parent) {
            command.option('-v, --version', 'output the application version');
        }
        if (!command.categories) {
            command.categories = {};
        }
        command.category = function (name) {
            var category = command.categories[name];
            if (!command.categories[name] || (command.categories[name]).stub && this.executingCmd) {
                category = command.categories[name] = new CtmCli(name, this);
                command.categories[name].stub = false;
                category.helpInformation = command.categoryHelpInformation;
                command.enableNestedCommands(category);
            }
            return category;
        };
        command.on('*', function () {
            var args = command.rawArgs.slice(0, 2);
            var raw = command.normalize(command.rawArgs.slice(2));
            var category = '*';
            if (raw.length > 0) {
                category = raw[0];
                args = args.concat(raw.slice(1));
            }
            var i, index;
            var targetCmd;
            var cat = command.categories[category];
            if (!cat) {
                index = command.searchCommand(category, command.commands);
                if (index !== -1) {
                    var cmd = command.commands[index];
                    targetCmd = require(cmd.filePath);
                    targetCmd.init.apply(command, [command]);
                    command.rawArgs[2] = cmd.name();
                    return this.parse(command.rawArgs);
                }
                cat = command.searchCategory(category);
            }
            var rootCat = cat;
            this.prompt = this.findPromptInArgs(args);
            for (i = 2; cat && i < args.length && args[i] !== '-h' && args[i] !== '--help'; i++) {
                index = command.searchCommand(args[i], cat.commands);
                if (index !== -1) {
                    targetCmd = cat.commands[index];
                    break;
                }
                index = command.searchCommand(constants_1.Constants.EMPTY_COMMAND_NAME, cat.commands);
                if (index !== -1) {
                    targetCmd = cat.commands[index];
                    args.splice(i, 0, constants_1.Constants.EMPTY_COMMAND_NAME);
                    break;
                }
                cat = cat.searchCategory(args[i]);
            }
            if (targetCmd) {
                if (i + 1 < args.length && (args[i + 1] === '-h' || args[i + 1] === '--help')) {
                    return targetCmd.commandHelpInformation();
                }
                this.executingCmd = true;
                if (!this.workaroundOnAsmSiteCommands(targetCmd, command)) {
                    targetCmd = require(targetCmd.filePath);
                    targetCmd.init(command);
                }
                cat = command.categories[rootCat.name()];
                return cat.parse(args);
            }
            if (!cat) {
                var toBlame = (i > 2) ? args[i - 1] : category;
                this.exit(1, 'error', ('\'' + toBlame + '\' is not a ctm command. See \'ctm help\'.').red.bold);
            }
            else {
                command.categoryHelpInformation.apply(cat, []);
            }
        });
    },
    findPromptInArgs: function (args) {
        for (var i = 0; i < args.length; i++) {
            if (args[i] === '-p' || args[i] === '--prompt') {
                return true;
            }
        }
        return false;
    },
    workaroundOnAsmSiteCommands: function (targetCmd, command) {
        if (path.basename(targetCmd.filePath).indexOf('site.') !== -1) {
            var siteCmdDir = path.dirname(targetCmd.filePath);
            var siteCmdFiles = utilsCore.getFiles(siteCmdDir, false);
            var filesToLoad = {};
            var sitePlugins = [];
            siteCmdFiles.forEach(function (f) {
                var basename = path.basename(f);
                if (basename.indexOf('site.') === 0) {
                    var nameWithoutExt = basename.substring(0, basename.lastIndexOf('.'));
                    var ext = path.extname(basename);
                    if (filesToLoad[nameWithoutExt]) {
                        if (ext === '.js') {
                            filesToLoad[nameWithoutExt] = f;
                        }
                    }
                    else {
                        filesToLoad[nameWithoutExt] = f;
                    }
                }
            });
            Object.keys(filesToLoad).forEach(function (f) {
                sitePlugins.push(require(filesToLoad[f]));
            });
            sitePlugins.forEach(function (plugin) {
                if (plugin.init) {
                    plugin.init(command);
                }
            });
            return true;
        }
        else {
            return false;
        }
    },
    command: function (name) {
        var args = name.split(/ +/);
        var cmd = new CtmCli(args.shift(), this);
        var caller;
        try {
            caller = callerId.getData();
        }
        catch (e) {
            log.verbose('Failed to get caller ID', e);
        }
        if (caller && caller.filePath) {
            cmd.filePath = caller.filePath;
        }
        cmd.helpInformation = cmd.commandHelpInformation;
        var index = this.searchCommand(cmd.name, this.commands);
        if (index !== -1) {
            this.commands[index] = cmd;
        }
        else {
            this.commands.push(cmd);
        }
        cmd.parseExpectedArgs(args);
        return cmd;
    },
    stringStartWith: function (s, search) {
        return s && s.indexOf && s.indexOf(search) === 0;
    },
    searchCommand: function (name, commands) {
        if (!commands || !name) {
            return -1;
        }
        for (var i = 0; i < commands.length; i++) {
            if (commands[i].name() === name) {
                return i;
            }
        }
        var found = -1;
        for (var i = 0; i < commands.length; i++) {
            if (this.stringStartWith(commands[i].name(), name)) {
                if (found === -1) {
                    found = i;
                }
                else {
                    return -1;
                }
            }
        }
        return found;
    },
    searchCategory: function (name) {
        if (!name) {
            return null;
        }
        var found = null;
        for (var cat in this.categories) {
            if (this.stringStartWith(cat, name)) {
                if (found) {
                    return null;
                }
                else {
                    found = this.categories[cat];
                }
            }
        }
        return found;
    },
    deprecatedDescription: function (text, newCommand) {
        return this.description(util.format('%s (deprecated. This command is deprecated and will be removed in a future version. Please use \"%s\" instead', text, newCommand));
    },
    detailedDescription: function (str) {
        if (0 === arguments.length) {
            return this._detailedDescription;
        }
        this._detailedDescription = str;
        return this;
    },
    checkNodeForPromptly: function () {
        var version = process.version.substr(1);
        var ver = version.split('.');
        var ver1num = parseInt(ver[0], 10);
        var ver2num = parseInt(ver[1], 10);
        if (ver1num < 6 || (ver1num === 6 && ver2num < 1)) {
            console.log('You need node.js v6.1.0 or higher to use prompt. Your version v' +
                version + ' won\'t work either.');
            this.exit(0);
        }
    },
    checkVersion: function () {
        var version = process.version;
        var ver = version.split('.');
        var ver1num = parseInt(ver[1], 10);
        var ver2num = parseInt(ver[2], 10);
        if (ver[0] === 'v0') {
            if (ver1num < 6 || (ver1num === 6 && ver2num < 15)) {
                throw new Error('You need node.js v0.6.15 or higher to run this code. Your version: ' +
                    version);
            }
            if (ver1num === 7 && ver2num <= 7) {
                throw new Error('You need node.js v0.6.15 or higher to run this code. Your version ' +
                    version + ' won\'t work either.');
            }
        }
    }
});
exports = module.exports = CtmCli;
//# sourceMappingURL=cli.js.map