'use strict';
var _ = require('underscore');
var fs = require('fs');
var path = require('path');
var utilsCore = require('./util/utilsCore');
var CmdLoader = (function () {
    function CmdLoader(topCmd, cmdMode) {
        this.topCmd = topCmd;
        this.cmdMode = cmdMode;
        this.cmdMetadataFile = path.join(__dirname, 'plugins.' + this.cmdMode + '.json');
        this.cmdBasePath = __dirname;
    }
    CmdLoader.prototype.harvestPlugins = function (topLevelOnly) {
        var self = this;
        var plugins = this._loadCmdsFromFolder(path.join(this.cmdBasePath, 'commands'), false);
        plugins.forEach(function (plugin) { plugin.init(self.topCmd); });
        if (!topLevelOnly) {
            var modePlugins = this._loadCmdsFromFolder(path.join(this.cmdBasePath, 'commands', this.cmdMode), true);
            modePlugins.forEach(function (plugin) { plugin.init(self.topCmd); });
        }
    };
    CmdLoader.prototype.harvestModules = function () {
        var self = this;
        var basePath = path.dirname(__filename);
        var walkPath = path.join(basePath, '../node_modules');
        var harvestPaths = [walkPath];
        while (path.basename(walkPath) === 'node_modules' && path.dirname(walkPath) !== 'npm') {
            var nextPath = path.join(walkPath, '..');
            if (nextPath === walkPath) {
                break;
            }
            harvestPaths.push(nextPath);
            walkPath = nextPath;
        }
        var modules = [];
        harvestPaths.forEach(function (harvestPath) {
            modules = modules.concat(self._loadCmdsFromNodeModules(harvestPath));
        });
        modules.forEach(function (module) {
            module.plugin.init(self.topCmd);
        });
    };
    CmdLoader.prototype.initFromCmdMetadata = function (CtmCli) {
        var self = this;
        var initCategory = function (category, parent) {
            function process(entity, entityParent) {
                var newEntity = new CtmCli(entity.name, entityParent);
                if (entity.description) {
                    newEntity._description = entity.description;
                }
                newEntity.description = function () {
                    return newEntity._description;
                };
                newEntity.fullName = function () {
                    return entity.fullName;
                };
                newEntity._usage = entity.usage;
                newEntity.usage = function () {
                    return newEntity._usage;
                };
                newEntity.filePath = entity.filePath ?
                    path.resolve(self.cmdBasePath, entity.filePath) : entity.filePath;
                newEntity.stub = true;
                if (entity.options) {
                    for (var o in entity.options) {
                        newEntity.option(entity.options[o].flags, entity.options[o].description);
                    }
                }
                newEntity.isCommandHidden = entity.isCommandHidden;
                return newEntity;
            }
            var newCategory = category;
            if (parent) {
                newCategory = process(category, parent);
            }
            for (var i = 0; i < category.commands.length; i++) {
                newCategory.commands[i] = (process(category.commands[i], newCategory));
            }
            if (!newCategory.categories) {
                newCategory.categories = {};
            }
            for (var j in category.categories) {
                newCategory.categories[j] = initCategory(category.categories[j], newCategory);
            }
            return newCategory;
        };
        var data = fs.readFileSync(this.cmdMetadataFile);
        var cachedPlugins = JSON.parse(data);
        var plugins = initCategory(cachedPlugins);
        this.topCmd.commands = plugins.commands;
        this.topCmd.categories = plugins.categories;
    };
    CmdLoader.prototype.saveCmdMetadata = function () {
        var metadate = this._serializeCategory(this.topCmd);
        fs.writeFileSync(this.cmdMetadataFile, JSON.stringify(metadate, null, 2));
    };
    CmdLoader.prototype.cmdMetadataExists = function () {
        return utilsCore.pathExistsSync(this.cmdMetadataFile);
    };
    CmdLoader.prototype._loadCmdsFromFolder = function (scanPath, recursively) {
        var results = utilsCore.getFiles(scanPath, recursively);
        results = results.filter(function (filePath) {
            var extname = path.extname(filePath);
            if (filePath.substring(0, 5) === 'tmp--') {
                return false;
            }
            else if (extname !== '.js' && extname !== '._js') {
                return false;
            }
            return true;
        });
        if (process.env.PRECOMPILE_STREAMLINE_FILES) {
            results = results.filter(function (filePath) {
                if (filePath.substring(filePath.length - 4) === '._js') {
                    return false;
                }
                return true;
            });
        }
        results = results.sort();
        results = results.filter(function (filePath) {
            return fs.statSync(filePath).isFile();
        });
        results = results.map(function (filePath) {
            return require(filePath);
        });
        results = results.filter(function (entry) {
            return entry.init !== undefined;
        });
        return results;
    };
    CmdLoader.prototype._loadCmdsFromNodeModules = function (scanPath) {
        var results = fs.readdirSync(scanPath);
        results = results.map(function (moduleName) {
            return {
                moduleName: moduleName,
                modulePath: path.join(scanPath, moduleName)
            };
        });
        results = results.filter(function (item) {
            try {
                item.moduleStat = fs.statSync(item.modulePath);
            }
            catch (error) {
                return false;
            }
            return item.moduleStat.isDirectory();
        });
        results = results.filter(function (item) {
            item.packagePath = path.join(item.modulePath, 'package.json');
            item.packageStat = utilsCore.pathExistsSync(item.packagePath) ? fs.statSync(item.packagePath) : undefined;
            return item.packageStat && item.packageStat.isFile();
        });
        results = results.filter(function (item) {
            try {
                item.packageInfo = JSON.parse(fs.readFileSync(item.packagePath));
                return item.packageInfo && item.packageInfo.plugins && item.packageInfo.plugins['ctm-cli'];
            }
            catch (err) {
                return false;
            }
        });
        results = this._flatten(results.map(function (item) {
            var plugins = item.packageInfo.plugins['ctm-cli'];
            if (!_.isArray(plugins)) {
                plugins = [plugins];
            }
            return plugins.map(function (relativePath) {
                return {
                    context: item,
                    pluginPath: path.join(item.modulePath, relativePath)
                };
            });
        }));
        results = results.filter(function (item) {
            item.plugin = require(item.pluginPath);
            return item.plugin.init;
        });
        return results;
    };
    CmdLoader.prototype._flatten = function (arrays) {
        var result = [];
        arrays.forEach(function (array) {
            result = result.concat(array);
        });
        return result;
    };
    CmdLoader.prototype._serializeIndividualEntity = function (entity) {
        var cmdOrCat = {};
        if (entity.name) {
            cmdOrCat.name = entity.name() ? entity.name() : '';
        }
        if (entity.description) {
            cmdOrCat.description = entity.description();
        }
        if (entity.fullName) {
            cmdOrCat.fullName = entity.fullName();
        }
        if (entity.usage) {
            cmdOrCat.usage = entity.usage();
        }
        if (entity.filePath) {
            cmdOrCat.filePath = path.relative(this.cmdBasePath, entity.filePath).split('\\').join('/');
        }
        if (entity.options) {
            cmdOrCat.options = entity.options;
        }
        cmdOrCat.isCommandHidden = entity.isCommandHidden;
        return cmdOrCat;
    };
    CmdLoader.prototype._serializeCategory = function (category) {
        var cat = this._serializeIndividualEntity(category);
        cat.commands = [];
        cat.categories = {};
        if (category.commands) {
            for (var i in category.commands) {
                cat.commands.push(this._serializeIndividualEntity(category.commands[i]));
            }
        }
        if (category.categories) {
            for (var j in category.categories) {
                var currentCategory = this._serializeCategory(category.categories[j]);
                cat.categories[currentCategory.name] = currentCategory;
            }
        }
        return cat;
    };
    return CmdLoader;
})();
exports.CmdLoader = CmdLoader;
//# sourceMappingURL=cmdLoader.js.map