"use strict";
var fs = require("fs");
var path = require("path");
var environments_1 = require("../../util/environments");
var promptData_1 = require("../../util/promptData");
var utilsCore = require("../../util/utilsCore");
var S = utilsCore.getLocaleString;
var environmentService;
function printEnvironments() {
    console.info(S("current environment: %s", environments_1.environments.getCurrentEnvironmentName()));
    console.info(S("environments: %s", JSON.stringify(environments_1.environments.getDisplayableEnvironments(), null, 2)));
}
function printEnvironment(envName, msg) {
    console.info(S(msg, envName));
    var env = environments_1.environments.getEnvironment(envName);
    console.info(S("%s: %s", envName, JSON.stringify(env.toDisplayable())));
}
function checkRootCertificateRequired() {
    if (!environments_1.environments.getRootCertificateRequired()) {
        console.warn(("ctm-cli is configured to accept self-signed certificates.\n").red.bold +
            "To accept only signed certificate you may run:\n" +
            "    ctm environment configure rootCertificateRequired true");
    }
}
function addEnvAction(name, args, description, option, action) {
    var cli = environmentService;
    cli = cli.command(name + " " + args);
    cli = cli.description(description);
    if (option)
        cli = cli.option(option[0], option[1]);
    cli = cli.execute(action);
    cli = cli.usage(args);
}
function init(cli) {
    environmentService = cli
        .category("environment")
        .description("Define and select the Control-M environment");
    addEnvAction("configure", "<setting-name> [setting-value]", "configure global environment settings", null, function (settingName, settingValue) {
        try {
            var msg = "";
            if (environments_1.environments.setSetting(settingName, settingValue)) {
                msg = "Setting \'%s\' was set to \'%s\'.";
            }
            else {
                msg = "Setting \'%s\' is already set to \'%s\'. No change was done.";
            }
            console.info(S(msg, settingName, settingValue));
        }
        catch (error) {
            cli.exit(1, "error", error);
        }
    });
    addEnvAction("show", " ", "display environments settings", null, printEnvironments);
    addEnvAction("set", "<env>", "select current environment", null, function (envName) {
        try {
            environments_1.environments.setCurrentEnvironment(envName);
            printEnvironments();
        }
        catch (error) {
            cli.exit(1, "error", error);
        }
    });
    addEnvAction("delete", "<env>", "delete an environment", null, function (envName) {
        try {
            environments_1.environments.deleteEnvironment(envName);
            printEnvironments();
        }
        catch (error) {
            cli.exit(1, "error", error);
        }
    });
    function doEnvUpdate(envName, propName, value, options, callback) {
        var getValue = function () {
            if (options.prompt && propName === 'password') {
                var promptDataArray = new Array();
                promptDataArray.push(new promptData_1.PromptData("Password", true));
                return cli.processPromisesArray(promptDataArray, utilsCore.promptWithValidate);
            }
            else {
                return Promise.resolve([value]);
            }
        };
        getValue().then(function (values) {
            try {
                if (!values[0])
                    cli.exit(1, "error", "missing required argument `value'");
                environments_1.environments.updateEnvironment(envName, propName, values[0]);
                printEnvironment(envName, "Environment \'%s\' was updated");
            }
            catch (error) {
                cli.exit(1, "error", error);
            }
        }).catch(function (error) { return cli.exit(1, "error", error); });
    }
    addEnvAction("update", "<env> <name> [value]", "update an environment property. the property <name> can be endPoint, user, password or additionalLoginHeader", ["-p, --prompt ", "Use prompt for this command"], doEnvUpdate);
    function doEnvAdd(envName, endPoint, user, password, options, callback) {
        var getPassword = function () {
            if (options.prompt) {
                var promptDataArray = new Array();
                promptDataArray.push(new promptData_1.PromptData("Password", true));
                return cli.processPromisesArray(promptDataArray, utilsCore.promptWithValidate);
            }
            else {
                return Promise.resolve([password]);
            }
        };
        getPassword().then(function (values) {
            try {
                if (!values[0])
                    cli.exit(1, "error", "missing required argument `password'");
                environments_1.environments.addEnvironment(envName, endPoint, user, values[0]);
                printEnvironment(envName, "Environment \'%s\' was created");
            }
            catch (error) {
                cli.exit(1, "error", error);
            }
        }).catch(function (error) { return cli.exit(1, "error", error); });
    }
    addEnvAction("add", "<env> <endPoint> <user> [password]", "add a new environment", ["-p, --prompt ", "Use prompt for this command"], doEnvAdd);
    addEnvAction("workbench::add", "[endPoint]", "add a workbench environment", null, function (endPoint) {
        try {
            environments_1.environments.addWorkbenchEnvironment(endPoint);
            printEnvironment("workbench", "Environment \'%s\' was created");
        }
        catch (error) {
            cli.exit(1, "error", error);
        }
    });
    addEnvAction("copy", "<env> <newEnv>", "duplicate an environment", null, function (envName, newEnvName) {
        try {
            environments_1.environments.copyEnvironment(envName, newEnvName);
            printEnvironment(newEnvName, S("Environment \'%s\' was copied to \'%s\'", envName));
        }
        catch (error) {
            cli.exit(1, "error", error);
        }
    });
    addEnvAction("load", "<enviromentsFile>", "load environments from file", null, function (environmentFile) {
        try {
            var pathEnvironmentsFile = path.resolve(environmentFile);
            if (!fs.existsSync(pathEnvironmentsFile)) {
                throw S("File %s not found", pathEnvironmentsFile);
            }
            environments_1.environments.loadFromFile(pathEnvironmentsFile);
            printEnvironments();
        }
        catch (error) {
            cli.exit(1, "error", error);
        }
    });
}
exports.init = init;
//# sourceMappingURL=environment_service.js.map