'use strict';
var _ = require('underscore');
var colors = require('colors');
var commander = require('commander');
var fs = require('fs');
var path = require('path');
var util = require('util');
var utilsCore = require('./util/utilsCore');
var pjson = require('../package.json');
var constants_1 = require("./util/constants");
function pad(str, width) {
    var len = Math.max(0, width - str.length);
    return str + Array(len + 1).join(' ');
}
function ExtendedCommand(name) {
    ExtendedCommand['super_'].call(this, name);
}
util.inherits(ExtendedCommand, commander.Command);
_.extend(ExtendedCommand.prototype, {
    fullName: function () {
        var name = this.name() ? this.name() : '';
        var scan = this.parent;
        while (scan && scan.parent !== undefined) {
            name = scan.name() ? scan.name() + ' ' + name : name;
            scan = scan.parent;
        }
        return name.replace(constants_1.Constants.EMPTY_COMMAND_NAME, '').trim();
    },
    usage: function (str) {
        var ret;
        if (str) {
            ret = commander.Command.prototype.usage.call(this, str);
        }
        else {
            ret = commander.Command.prototype.usage.call(this);
            ret = ret.replace(/,/g, ' ');
        }
        return ret;
    },
    helpInformation: function () {
        var self = this;
        if (!self.parent) {
            var raw = self.normalize(process.argv.slice(2));
            var packagePath = path.join(__dirname, '../package.json');
            var packageInfo = JSON.parse(fs.readFileSync(packagePath));
            if (raw.indexOf('-v') >= 0) {
                console.log(packageInfo.version);
            }
            else if (raw.indexOf('--version') >= 0) {
                console.log(util.format('%s (node: %s)', packageInfo.version, process.versions.node));
            }
            else {
                self.setupCommandLogFormat(true);
                self.helpCommands();
                self.helpCategoriesSummary(self.showMore());
                self.helpOptions();
            }
        }
        else {
            self.output.help(self.description());
            self.output.help('');
            self.output.help('Usage:', self.fullName() + ' ' + self.usage());
            self.helpOptions();
        }
        return '';
    },
    showMore: function () {
        var raw = this.normalize(process.argv.slice(2));
        return raw.indexOf('--help') >= 0 || raw.indexOf('-h') >= 0;
    },
    categoryHelpInformation: function () {
        var self = this;
        this.output.help(self.description());
        self.helpCommands();
        self.helpCategories(-1);
        return '';
    },
    commandHelpInformation: function () {
        var self = this;
        if (self._detailedDescription) {
            this.output.help(self.detailedDescription());
        }
        else {
            this.output.help(self.description());
        }
        this.output.help('');
        this.output.help('Usage:', self.fullName() + ' ' + self.usage());
        self.helpOptions();
        return '';
    },
    helpJSON: function () {
        var self = this;
        var result = _.tap({}, function (res) {
            if (_.isNull(self.parent) || _.isUndefined(self.parent)) {
                res.name = pjson.name;
                res.description = pjson.description;
                res.author = pjson.author;
                res.version = pjson.version;
                res.contributors = pjson.contributors;
                res.homepage = pjson.homepage;
                res.licenses = pjson.licenses;
            }
            else {
                res.name = self.fullName();
                res.description = self.description();
            }
            res.usage = self.usage();
        });
        if (this.categories && Object.keys(this.categories).length > 0) {
            result.categories = {};
            for (var name in this.categories) {
                result.categories[name] = this.categories[name].helpJSON();
            }
        }
        if (this.commands && this.commands.length > 0) {
            result.commands = [];
            this.commands.forEach(function (cmd) {
                var command = {
                    name: cmd.fullName(),
                    description: cmd.description(),
                    options: cmd.options,
                    usage: cmd.usage()
                };
                result.commands.push(command);
            });
        }
        return result;
    },
    helpCategories: function (levels) {
        for (var name in this.categories) {
            var cat = this.categories[name];
            this.output.help('');
            this.output.help(cat.description().cyan);
            if (levels === -1 || levels > 0) {
                for (var index in cat.commands) {
                    var cmd = cat.commands[index];
                    this.output.help(' ', cmd.fullName() + ' ' + cmd.usage());
                }
                cat.helpCategories(levels !== -1 ? --levels : -1);
            }
            else {
                this.output.help(' ', cat.fullName());
            }
        }
    },
    helpCategoriesSummary: function (showMore) {
        var self = this;
        var categories = [];
        function scan(parent, levels, each) {
            for (var name in parent.categories) {
                var cat = parent.categories[name];
                each(cat);
                if (levels === -1 || levels > 0) {
                    scan(cat, levels !== -1 ? --levels : -1, each);
                }
            }
        }
        scan(this, showMore ? -1 : 0, function (cat) { categories.push(cat); });
        var maxLength = 14;
        categories.sort(function (a, b) {
            return (a.fullName() < b.fullName()) ? -1 : (a.fullName() > b.fullName()) ? 1 : 0;
        });
        categories.forEach(function (cat) {
            if (maxLength < cat.fullName().length)
                maxLength = cat.fullName().length;
        });
        self.output.help('');
        self.output.help('Commands:');
        categories.forEach(function (cat) {
            var name = cat.fullName();
            while (name.length < maxLength) {
                name += ' ';
            }
            var description = cat.description();
            if (!description)
                description = "";
            self.output.help('  ' + name + ' ' + description.cyan);
        });
    },
    helpCommands: function () {
        var self = this;
        this.commands.forEach(function (cmd) {
            if (!cmd.isCommandHidden || process.env.CTM_DEBUG_FUTURE_API) {
                self.output.help('');
                self.output.help(cmd.description().cyan);
                self.output.help(' ', cmd.fullName() + ' ' + cmd.usage());
            }
        });
        self.helpCommandsOptions();
    },
    helpCommandsOptions: function () {
        var self = this;
        var optionsConsolidation = {};
        var width = 0;
        this.commands.forEach(function (cmd) {
            cmd.options.forEach(function (opt) {
                optionsConsolidation[opt.flags] = opt;
            });
            width = Math.max(width, cmd.largestOptionLength());
        });
        if (Object.keys(optionsConsolidation).length > 0) {
            self.output.help('');
            self.output.help('Usage: [command] [options]');
            self.output.help('');
            self.output.help('Options:');
            for (var optKey in optionsConsolidation) {
                var opt = optionsConsolidation[optKey];
                self.output.help(' ', pad(opt.flags, width) + '  ' + opt.description);
            }
        }
    },
    helpOptions: function () {
        var self = this;
        self.output.help('');
        self.output.help('Options:');
        self.optionHelp().split('\n').forEach(function (line) { self.output.help(' ', line); });
    },
    fileRelatedOption: function (flags, description) {
        this.option(flags, description);
        this.options[this.options.length - 1].fileRelatedOption = true;
        return this;
    },
    option: function (flags, description, fn, defaultValue) {
        var self = this;
        var option = new commander.Option(flags, description);
        self.options = self.options.filter(function (o) {
            return o.name() !== option.name() || o.long !== option.long;
        });
        var oname = option.name();
        var name = utilsCore.camelcase(oname);
        if (!self.optionValues) {
            self.optionValues = {};
        }
        if ('function' !== typeof fn) {
            defaultValue = fn;
            fn = null;
        }
        if (false === option.bool || option.optional || option.required) {
            if (false === option.bool)
                defaultValue = true;
            if (undefined !== defaultValue)
                self.optionValues[name] = defaultValue;
        }
        this.options.push(option);
        this.on(oname, function (val) {
            if (val && fn)
                val = fn(val);
            if ('boolean' === typeof self.optionValues[name] || 'undefined' === typeof self.optionValues[name]) {
                if (!val) {
                    self.optionValues[name] = option.bool ? defaultValue || true : false;
                }
                else {
                    self.optionValues[name] = val;
                }
            }
            else if (val) {
                self.optionValues[name] = val;
            }
        });
        return this;
    }
});
module.exports = ExtendedCommand;
//# sourceMappingURL=extendedcommand.js.map