var request = require('request');
var lockfile = require("lockfile");
var perfy = require('perfy');
var targz = require('targz');
var fs = require("fs");
var os = require("os");
var path = require("path");
var log = require('../util/logging');
var utilsCore = require('../util/utilsCore');
var environments_1 = require("../util/environments");
var buildInfo_1 = require("./buildInfo");
var tracer_1 = require("./tracer");
var version_1 = require("./version");
var ModuleLoader = (function () {
    function ModuleLoader() {
        tracer_1.Tracer.setFromArgv(process.argv);
        this.defaultVersion = require('../../package.json').version;
        tracer_1.Tracer.log("defaultVersion: " + this.defaultVersion);
        this.installedModuleRootPath = path.dirname(require.resolve('../../package.json'));
        tracer_1.Tracer.log("installedModuleRootPath: " + this.installedModuleRootPath);
        if (this.installedModuleRootPath !== this.getModulePath(this.defaultVersion))
            tracer_1.Tracer.log("default client module is different from installed module because an update; it pash is " + this.getModulePath(this.defaultVersion));
    }
    ModuleLoader.prototype.load = function () {
        var _this = this;
        var endPoint = ModuleLoader.getEndPoint();
        if (!endPoint) {
            tracer_1.Tracer.log("there is no endPoint, run default module with version  " + this.defaultVersion);
            this.runModule(this.defaultVersion);
            return;
        }
        tracer_1.Tracer.log("endPoint is set to " + endPoint);
        if (!environments_1.environments.getRootCertificateRequired()) {
            tracer_1.Tracer.log("authorize using non-authorize certificate because rootCertificateRequired is set to false or missed.");
            process.env.NODE_TLS_REJECT_UNAUTHORIZED = "0";
        }
        else {
            tracer_1.Tracer.log("reject using non-authorize certificate because rootCertificateRequired is set to true.");
        }
        tracer_1.Tracer.log("getting build info from endPoint " + endPoint);
        this.getServerBuildInfo(endPoint).then(function (serverBuildInfo) {
            tracer_1.Tracer.log("check if there is module with this build info " + JSON.stringify(serverBuildInfo));
            var isVersionTheSame = _this.isModuleExisted(serverBuildInfo);
            if (isVersionTheSame && _this.isModuleUpdated(serverBuildInfo)) {
                tracer_1.Tracer.log('module is found and has the same build time as server, run it...');
                _this.runModule(serverBuildInfo.version);
                return;
            }
            var localBuildTime = _this.getLocalBuildTime(serverBuildInfo.version);
            if (!isVersionTheSame)
                log.info("Server version " + serverBuildInfo.version + " is different from client version " + _this.defaultVersion);
            else
                log.info("Server build time " + serverBuildInfo.time + " is different from client built time " + localBuildTime);
            var lock_file = path.join(utilsCore.ctmDir(), '.lockfile');
            tracer_1.Tracer.log("taking lock " + lock_file + " ...");
            lockfile.lock(lock_file, {
                wait: 300000,
                stale: 600000
            }, function (error) {
                if (error) {
                    log.error("fail to take lock: " + lock_file + " -- " + error);
                    return;
                }
                if (_this.isModuleExistedAndUpdated(serverBuildInfo)) {
                    tracer_1.Tracer.log('client module is found after the lock was taken, run it...');
                    lockfile.unlock(lock_file, function () { });
                    _this.runModule(serverBuildInfo.version);
                    return;
                }
                tracer_1.Tracer.log("module with the server build info is not found after taking lock, download it from endpoint");
                log.info("Starting download of server version " + serverBuildInfo.version + " from " + endPoint);
                perfy.start('download');
                _this.downloadModule(endPoint)
                    .then(function (filename) {
                    log.info("Download finished in " + perfy.end('download').time + "  sec.");
                    _this.decompress(filename, serverBuildInfo.version)
                        .then(function () {
                        _this.writeLocalBuildInfo(serverBuildInfo);
                        lockfile.unlock(lock_file, function () { });
                        _this.runModule(serverBuildInfo.version);
                    })
                        .catch(function (error) {
                        log.error("install failed with error " + error.message);
                    });
                })
                    .catch(function (error) {
                    log.error("download failed with error " + error.message);
                });
            });
        });
    };
    ModuleLoader.prototype.getModulePath = function (version) {
        var homeDir = utilsCore.ctmDir();
        var moduleName = "ctm-cli-" + version;
        var modulePath = path.join(homeDir, 'node_modules', moduleName);
        if (!fs.existsSync(modulePath) && version === this.defaultVersion) {
            return this.installedModuleRootPath;
        }
        return modulePath;
    };
    ModuleLoader.getEndPoint = function () {
        if (!environments_1.environments.getCurrentEnvironmentName()) {
            return null;
        }
        var env = environments_1.environments.getCurrentEnvironment();
        var endPoint = env.endPoint;
        var i = process.argv.indexOf('-e');
        if (i !== -1 && i + 1 < process.argv.length) {
            var envName = process.argv[i + 1];
            var env_1 = environments_1.environments.getEnvironment(envName);
            if (env_1)
                endPoint = env_1.endPoint;
        }
        return endPoint;
    };
    ModuleLoader.prototype.doRequest = function (url) {
        return new Promise(function (resolve, reject) {
            var options = {
                url: url,
                method: 'GET',
                timeout: 3000
            };
            perfy.start('doRequest');
            request(options, function (error, response, body) {
                tracer_1.Tracer.log("request to " + url + " finished in " + perfy.end('doRequest').time + " sec.");
                if (error)
                    reject(error);
                if (body)
                    resolve(body);
            });
        });
    };
    ModuleLoader.prototype.isModuleExisted = function (serverBuildInfo) {
        var modulePath = this.getModulePath(serverBuildInfo.version);
        return fs.existsSync(modulePath);
    };
    ModuleLoader.prototype.isModuleUpdated = function (serverBuildInfo) {
        if (serverBuildInfo.version === this.defaultVersion)
            return true;
        if (serverBuildInfo.time === null) {
            tracer_1.Tracer.log("missing server " + serverBuildInfo.version + " build time, no update is needed");
            return true;
        }
        var localBuildTime = this.getLocalBuildTime(serverBuildInfo.version);
        if (localBuildTime === null) {
            tracer_1.Tracer.log("missing client " + serverBuildInfo.version + " build time, no update is needed");
            return true;
        }
        if (serverBuildInfo.time === localBuildTime) {
            tracer_1.Tracer.log("server build time " + serverBuildInfo.time + " is matched the client module build time " + localBuildTime + ", no update is needed");
            return true;
        }
        else {
            tracer_1.Tracer.log("server build time " + serverBuildInfo.time + " does not match to the client module build time " + localBuildTime + " --- update is needed");
            return false;
        }
    };
    ModuleLoader.prototype.isModuleExistedAndUpdated = function (serverBuildInfo) {
        return this.isModuleExisted(serverBuildInfo) && this.isModuleUpdated(serverBuildInfo);
    };
    ModuleLoader.prototype.downloadModule = function (endPoint) {
        var url = endPoint + "/ctm-cli.tgz";
        var dest = path.join(os.tmpdir(), 'ctm-cli.tgz');
        return new Promise(function (resolve, reject) {
            var option = {
                url: url
            };
            request(url, option)
                .on('error', function (error) {
                reject(error);
            })
                .pipe(fs.createWriteStream(dest))
                .on('finish', function () {
                resolve(dest);
            });
        });
    };
    ModuleLoader.prototype.decompress = function (sourceFilname, version) {
        var modulePath = this.getModulePath(version);
        if (modulePath === this.installedModuleRootPath) {
            modulePath = path.join(utilsCore.ctmDir(), 'node_modules', "ctm-cli-" + version);
        }
        var dirname = path.dirname(modulePath);
        var moduleName = path.basename(modulePath);
        log.info("Installing version " + version + " to " + modulePath);
        perfy.start('decompress');
        return new Promise(function (resolve, reject) {
            return targz.decompress({
                src: sourceFilname,
                dest: dirname,
                tar: {
                    map: function (header) {
                        header.name = header.name.replace('package/', moduleName + "/");
                        return header;
                    }
                }
            }, function (err) {
                if (err) {
                    reject(err);
                }
                else {
                    log.info('Install finished in ' + perfy.end('decompress').time + ' sec.');
                    resolve();
                }
            });
        });
    };
    ModuleLoader.prototype.runModule = function (moduleVersion) {
        tracer_1.Tracer.log("run module with version " + moduleVersion);
        var modulePath = this.getModulePath(moduleVersion);
        var entryPoint;
        if (new version_1.Version(moduleVersion).compare(new version_1.Version('9.18.2')) <= 0) {
            entryPoint = path.join('bin', 'ctm');
            tracer_1.Tracer.log("use module entry point " + entryPoint + " because module version " + moduleVersion + " <= 9.18.2");
        }
        else {
            entryPoint = path.join('bin', 'ctm-main');
            tracer_1.Tracer.log("module entry point is " + entryPoint);
        }
        var moduleEntryPointFullPath = path.join(modulePath, entryPoint);
        tracer_1.Tracer.log("run module entry point from " + moduleEntryPointFullPath);
        require(moduleEntryPointFullPath);
    };
    ModuleLoader.prototype.getServerBuildInfo = function (endPoint) {
        var _this = this;
        var buildInfoUrl = endPoint + "/build_time.txt";
        return this.doRequest(buildInfoUrl).then(function (buildInfoText) {
            var buildInfo = buildInfo_1.BuildInfo.parseBuildInfo(buildInfoText);
            if (buildInfo === null) {
                tracer_1.Tracer.log("fail to parse server build info: " + buildInfoText + ", will use default version: " + _this.defaultVersion);
                return {
                    time: null,
                    version: _this.defaultVersion
                };
            }
            tracer_1.Tracer.log("server build info: " + buildInfo);
            return buildInfo;
        }).catch(function (error) {
            tracer_1.Tracer.log("fail to get server build info, error " + error.message + ", there is no connection to endpoint " + endPoint + ", use default version " + _this.defaultVersion);
            return {
                time: null,
                version: _this.defaultVersion
            };
        });
    };
    ModuleLoader.prototype.writeLocalBuildInfo = function (buildInfo) {
        var modulePath = this.getModulePath(buildInfo.version);
        var moduleBuildInfoPath = modulePath + '/build_time.txt';
        var data = buildInfo.toString();
        fs.writeFileSync(moduleBuildInfoPath, data);
    };
    ModuleLoader.prototype.getLocalBuildTime = function (version) {
        var modulePath = this.getModulePath(version);
        var moduleBuildInfoPath = modulePath + '/build_time.txt';
        if (!fs.existsSync(moduleBuildInfoPath)) {
            tracer_1.Tracer.log("build_time.txt is not found in " + moduleBuildInfoPath + ", for version " + version);
            return null;
        }
        var data = fs.readFileSync(moduleBuildInfoPath);
        var bi = buildInfo_1.BuildInfo.parseBuildInfo(data.toString());
        return bi ? bi.time : null;
    };
    return ModuleLoader;
})();
exports.ModuleLoader = ModuleLoader;
//# sourceMappingURL=moduleLoader.js.map