var configuration_1 = require("../../lib/util/configuration");
var path = require("path");
var utilsCore = require("./utilsCore");
var S = utilsCore.getLocaleString;
var IEnvironment = (function () {
    function IEnvironment(endPoint, user, password, additionalLoginHeader, needVerification) {
        if (needVerification === void 0) { needVerification = true; }
        this.endPoint = endPoint;
        this.user = user;
        this.password = password;
        this.additionalLoginHeader = additionalLoginHeader;
        this.endPoint = endPoint;
        this.user = user;
        if (password !== undefined) {
            this.password = password;
        }
        if (needVerification) {
            this.verify();
        }
    }
    Object.defineProperty(IEnvironment.prototype, "decriptedPassword", {
        get: function () {
            return utilsCore.decrypt(this.password);
        },
        enumerable: true,
        configurable: true
    });
    IEnvironment.prototype.verify = function () {
        if (this.endPoint) {
            this.endPoint = this.endPoint.replace(/\/+$/, "");
            if (!this.endPoint.match(/^https/i)
                && !this.endPoint.match(/^http:\/\/localhost\W/i)
                && !this.endPoint.match(/^http:\/\/127\.\d+\.\d+\.\d+\W/i)
                && !this.endPoint.match(/^http:\/\/\[::1\]\W/i)) {
                throw S("The endpoint \'%s\' is not secured." +
                    "Only secured endpoints that begin with \'https\' are supported.", this.endPoint);
            }
        }
    };
    IEnvironment.prototype.update = function (nameProp, value) {
        var _this = this;
        if (!this.hasOwnProperty(nameProp)) {
            var availableProps = Object.keys(this).filter(function (prop) { return typeof _this[prop] !== "function"; });
            throw S("Environment properties are [%s]. \'%s\' is illegal ", availableProps.join(", "), nameProp);
        }
        if (nameProp === "endPoint" && value.indexOf("http") !== 0) {
            throw S("Environment \'endPoint\' must be in the format of http URL");
        }
        if (nameProp === "password") {
            this["password"] = utilsCore.encrypt(value);
        }
        else {
            this[nameProp] = value;
        }
        this.verify();
    };
    IEnvironment.prototype.toDisplayable = function () {
        return new IEnvironment(this.endPoint, this.user, undefined, undefined, false);
    };
    return IEnvironment;
})();
exports.IEnvironment = IEnvironment;
var EnvironmentsSettings = (function () {
    function EnvironmentsSettings() {
        this.environments = {};
        this.loadEnvironmentFromFile();
        if (!this.getRootCertificateRequired()) {
            process.env.NODE_TLS_REJECT_UNAUTHORIZED = 0;
        }
    }
    EnvironmentsSettings.getEnvironmentsFile = function () {
        return path.join(utilsCore.ctmDir(), "env.json");
    };
    EnvironmentsSettings.prototype.addWorkbenchEnvironment = function (endPoint) {
        if (!endPoint) {
            endPoint = "https://localhost:8443/automation-api";
        }
        this.addEnvironment("workbench", endPoint, "workbench", "workbench");
    };
    EnvironmentsSettings.prototype.setSetting = function (settingName, settingValue) {
        var config = configuration_1.Configuration.getCliConfigurations();
        if (config[settingName] === settingValue) {
            return false;
        }
        config[settingName] = settingValue;
        configuration_1.Configuration.saveCliConfigurations(config);
        return true;
    };
    EnvironmentsSettings.prototype.getSettings = function (settingName, defaultValue) {
        var config = configuration_1.Configuration.getCliConfigurations();
        var result = config[settingName];
        return result === undefined ? defaultValue : result;
    };
    EnvironmentsSettings.prototype.getRootCertificateRequired = function () {
        return this.getSettings("rootCertificateRequired", "false") === "true";
    };
    EnvironmentsSettings.prototype.fixCurrentEnv = function () {
        var validCurrentEnv = this.environments && this.currentEnv && this.environments[this.currentEnv];
        if (!validCurrentEnv) {
            this.currentEnv = "";
        }
    };
    EnvironmentsSettings.prototype.getEnvironment = function (name) {
        return this.environments[name];
    };
    EnvironmentsSettings.prototype.deleteEnvironment = function (name) {
        var env = this.getEnvironment(name);
        if (!env) {
            throw S("Environment \'%s\' not found", name);
        }
        delete this.environments[name];
        this.fixCurrentEnv();
        this.saveEnvironmentToFile();
    };
    EnvironmentsSettings.prototype.findEnvironmentName = function (environment) {
        for (var name_1 in this.environments) {
            if (this.environments[name_1] === environment) {
                return name_1;
            }
        }
        return null;
    };
    EnvironmentsSettings.prototype.getCurrentEnvironmentName = function () {
        return this.currentEnv;
    };
    EnvironmentsSettings.prototype.getCurrentEnvironment = function () {
        return this.getEnvironment(this.currentEnv);
    };
    EnvironmentsSettings.prototype.setCurrentEnvironment = function (name) {
        var env = this.getEnvironment(name);
        if (!env) {
            throw S("Environment \'%s\' not found", name);
        }
        this.currentEnv = name;
        this.saveEnvironmentToFile();
    };
    EnvironmentsSettings.prototype.updateEnvironment = function (envName, nameProp, value) {
        var env = this.getEnvironment(envName) || new IEnvironment("", "", "");
        env.update(nameProp, value);
        this.environments[envName] = env;
        this.saveEnvironmentToFile();
    };
    EnvironmentsSettings.prototype.addEnvironment = function (envName, endPoint, user, password) {
        if (this.getEnvironment(envName)) {
            throw S("Environment \'%s\' already exists", envName);
        }
        this.environments[envName] = new IEnvironment(endPoint, user, utilsCore.encrypt(password));
        var validCurrentEnv = this.environments && this.currentEnv && this.environments[this.currentEnv];
        if (!validCurrentEnv) {
            this.setCurrentEnvironment(envName);
        }
        this.saveEnvironmentToFile();
    };
    EnvironmentsSettings.prototype.copyEnvironment = function (envName, newEnvName) {
        if (this.getEnvironment(newEnvName)) {
            throw S("Environment \'%s\' already exists", newEnvName);
        }
        var existing = this.getEnvironment(envName);
        if (!existing) {
            throw S("Environment \'%s\' cannot be found", envName);
        }
        this.environments[newEnvName] = new IEnvironment(existing.endPoint, existing.user, existing.password);
        this.saveEnvironmentToFile();
    };
    EnvironmentsSettings.prototype.loadEnvironmentValues = function (envValues, needVerification) {
        var _this = this;
        if (needVerification === void 0) { needVerification = true; }
        var enecrypted = utilsCore.pathExistsSync(utilsCore.getEncriptionKeyFilePath());
        if (envValues) {
            if (needVerification) {
                Object.keys(envValues.environments || {}).forEach(function (name) {
                    var envData = envValues.environments[name];
                    var env = new IEnvironment("", "", "");
                    Object.keys(envData).forEach(function (p) { return env[p] = envData[p]; });
                    env.verify();
                });
            }
            Object.keys(envValues.environments || {}).forEach(function (name) {
                if (!_this.environments[name]) {
                    _this.environments[name] = new IEnvironment("", "", "");
                }
                var envData = envValues.environments[name];
                var env = _this.environments[name];
                if (!enecrypted) {
                    if (envData["password"]) {
                        envData["password"] = utilsCore.encrypt(envData["password"]);
                    }
                }
                Object.keys(envData).forEach(function (p) { env[p] = envData[p]; });
            });
            if (envValues.currentEnv) {
                this.currentEnv = envValues.currentEnv;
            }
        }
        this.fixCurrentEnv();
        if (!enecrypted) {
            this.saveEnvironmentToFile();
        }
    };
    EnvironmentsSettings.prototype.loadFromFile = function (envFilePath) {
        var envValues = utilsCore.readJsonFile(envFilePath);
        Object.keys(envValues.environments || {}).forEach(function (name) {
            if (envValues.environments[name]["password"]) {
                envValues.environments[name]["password"] = utilsCore.encrypt(envValues.environments[name]["password"]);
            }
        });
        this.loadEnvironmentValues(envValues);
        this.saveEnvironmentToFile();
    };
    EnvironmentsSettings.prototype.getDisplayableEnvironments = function () {
        var _this = this;
        var environmentsForDisplay = {};
        Object.keys(this.environments).forEach(function (envName) {
            var env = _this.environments[envName];
            environmentsForDisplay[envName] = env.toDisplayable();
        });
        return environmentsForDisplay;
    };
    EnvironmentsSettings.prototype.loadEnvironmentFromFile = function () {
        var envFileValues = utilsCore.readJsonFile(EnvironmentsSettings.getEnvironmentsFile());
        this.loadEnvironmentValues(envFileValues, false);
    };
    EnvironmentsSettings.prototype.saveEnvironmentToFile = function () {
        utilsCore.writeJsonFile(this, EnvironmentsSettings.getEnvironmentsFile());
    };
    return EnvironmentsSettings;
})();
exports.EnvironmentsSettings = EnvironmentsSettings;
exports.environments = new EnvironmentsSettings();
//# sourceMappingURL=environments.js.map