import random
import time
import argparse

import asyncio

from traderion2.bot import TraderionBot
from traderion2.strategies.clients import Clients
from traderion2.strategies.ema import CrossingEMAs
from traderion2.strategies.macd import Macd
from traderion2.strategies.rsi import Rsi
from traderion2.strategies.cover import Cover
from traderion2.strategies.dev import StarterTemplate

# Direction constants
BEARISH_CROSS, BULLISH_CROSS = (0, 1)

CROSSING_EMA = 0
MACD = 1
RSI = 2
CLIENTS = 3
COVER = 4
DEV = 5

STRATEGY_MAP = {
    CROSSING_EMA: CrossingEMAs,
    MACD: Macd,
    RSI: Rsi,
    CLIENTS: Clients,
    COVER: Cover,
    DEV: StarterTemplate,
}


class CompositeBot(TraderionBot):
    """
    The CompositeBot class manages and executes multiple strategies.

    It requires a strategy_map argument during instantiation. The str_map is a nested dictionary structured as follows: \
        keys are swift_ids, each mapping to a dictionary where keys are strategy names, and values are dictionaries \
        containing parameters for the respective strategy. The CompositeBot calls the run method of all provided \
        strategies every second, ensuring synchronized execution of all strategies.

    Example:

        str_map = {
        4092: {
            MACD: {'target_amount': 100, 'small_ema_period': 5, 'big_ema_period': 20, 'signal_ema_period': 9},
            COVER: {'position_limit': 100},
            DEV: {},
            CLIENTS: {'client_quote_spread': lambda: random.randint(-5, 4), 'own_quote_spread': lambda: -22},
            CROSSING_EMA: {'target_amount': 100, 'small_ema_period': 5, 'big_ema_period': 20},
            RSI: {'target_amount': 100, 'period': 14, 'overbought_threshold': 70, 'oversold_threshold': 30},
        },
        4093: {
            CLIENTS: {'client_quote_spread': lambda: random.randint(-5, 4), 'own_quote_spread': lambda: -22},
            COVER: {'position_limit': 100},
            CROSSING_EMA: {'target_amount': 100, 'small_ema_period': 5, 'big_ema_period': 20},
            RSI: {'target_amount': 100, 'period': 65, 'overbought_threshold': 70, 'oversold_threshold': 30},
        }
    }
    bot = CompositeBot('USERNAME', 'PASSWORD', 9039450140436196, strategy_map)
    """

    def __init__(self, username, password, room_id, strategy_map=None, loop_sleep=1, show_info=False):
        super().__init__(username, password, room_id, loop_sleep, show_info)
        self.strategies = []

        for swift_id, strategies_pack in strategy_map.items():
            if strategies_pack is None or not isinstance(strategies_pack, dict):
                raise Exception("Provide a dictionary of strategy instances")
            else:
                for strategy_type, options in strategies_pack.items():
                    strategy_ctor = STRATEGY_MAP[strategy_type]
                    strategy = strategy_ctor(self.api, swift_id, options)
                    self.strategies.append(strategy)

    async def on_position_change(self, old_position, new_position):
        for strategy in self.strategies:
            await strategy.on_position_change(new_position)

    async def on_market_price_chart_updated(self, swift_id, new_price_curve):
        price_curves = self.api.get_price_curves()
        formated_price_curves = dict()

        for l_swift_id, points in price_curves.items():
            formated_price_curves[l_swift_id] = [float(point['price']) for point in points]

        for strategy in self.strategies:

            if strategy.swift_id == swift_id:
                await strategy.on_market_price_chart_updated(
                    formated_price_curves[strategy.swift_id]
                )

    async def on_eb_depth_change(self, swift_id, direction, old_depth, new_depth):
        for strategy in self.strategies:
            if strategy.swift_id == swift_id:
                await strategy.on_eb_depth_change(swift_id, direction, old_depth, new_depth)

    async def on_deal_done(self, swift_id, deal):
        for strategy in self.strategies:
            if strategy.swift_id == swift_id:
                await strategy.on_deal_done(
                    deal
                )

    async def on_new_client_call(self, client_call):
        swift_id = client_call['swift_id']

        for strategy in self.strategies:
            if strategy.swift_id == swift_id:
                await strategy.on_new_client_call(client_call)

    async def on_client_call_accepted(self, client_call):
        swift_id = client_call['swift_id']

        for strategy in self.strategies:
            if strategy.swift_id == swift_id:
                await strategy.on_client_call_accepted(client_call)

    async def on_client_reject(self, client_call):
        swift_id = client_call['swift_id']

        for strategy in self.strategies:
            if strategy.swift_id == swift_id:
                await strategy.on_client_reject(client_call)

    async def main_loop(self):
        for strategy in self.strategies:
            await strategy.run()

    def log_info(self):
        while True:
            s = ''
            for strategy in self.strategies:
                s += strategy.log_info() + '\n'

            clear_lines = '\033[F' * len(self.strategies)
            print(s, end=clear_lines, flush=True)
            time.sleep(1)


if __name__ == '__main__':
    """
    CROSSING_EMA Parameters
        :param target_amount: Amount in shares or millions.
        :param small_ema_period: The window of time for the small(signal) ema
        :param big_ema_period: The window of time for the big(base) ema

    MACD Parameters
        :param target_amount: Amount in shares or millions.
        :param small_ema_period: The window of time for the small ema
        :param big_ema_period: The window of time for the big ema
        :param signal_ema_period: The window of time for the signal ema

    RSI Parameters
        :param target_amount: Amount in shares or millions.
        :param period: The window of time for the ema 
        :param overbought_threshold: The value of RSI that signals selling
        :param oversold_threshold: The value of RSI that signals buying

    CLIENTS Parameters
        :param client_quote_spread: A function that returns the spread you are willing to accept from the client
        :param own_quote_spread: A function that returns the spread you are going to take when quoting to the client

    COVER Parameters
        :param position_limit: The max amount at risk. Absolute value. Eg 40 000 shares means you the bot will start 
            selling when position amount is greater than 40 000 shares or start buying when the position is less than 
            -40 000 shares  

    """

    parser = argparse.ArgumentParser(description="Boolean argument")
    parser.add_argument("--info", action="store_true", help="Display basic information about the room")

    args = parser.parse_args()

    str_map = {
        4172: {
            MACD: {'target_amount': 100, 'small_ema_period': 5, 'big_ema_period': 20, 'signal_ema_period': 9},
            COVER: {'position_limit': 100},
            DEV: {},
            CLIENTS: {'client_quote_spread': lambda: random.randint(-5, 4), 'own_quote_spread': lambda: -22},
            CROSSING_EMA: {'target_amount': 100, 'small_ema_period': 5, 'big_ema_period': 20},
            RSI: {'target_amount': 100, 'period': 14, 'overbought_threshold': 70, 'oversold_threshold': 30},
        },
        4173: {
            CLIENTS: {'client_quote_spread': lambda: random.randint(-5, 4), 'own_quote_spread': lambda: -22},
            COVER: {'position_limit': 100},
            CROSSING_EMA: {'target_amount': 100, 'small_ema_period': 5, 'big_ema_period': 20},
            RSI: {'target_amount': 100, 'period': 65, 'overbought_threshold': 70, 'oversold_threshold': 30},
        }
    } if args.info is False else {}

    bot = CompositeBot('USERNAME', 'PASSWORD', 1338201539833342, str_map, show_info=args.info)

    loop = asyncio.get_event_loop()
    loop.run_until_complete(asyncio.gather(bot.run()))
