class Base:

    def __init__(self, api, swift_id, options):
        if not isinstance(options, dict):
            raise Exception(f"Options must be a dict")

        self.api = api
        self.swift_id = swift_id

    async def on_position_change(self, new_position):
        """Called when the bot position changes. Override in subclass."""
        pass

    async def on_market_price_chart_updated(self, new_price_curve):
        """Called when the price curve changes. Override in subclass."""
        pass

    async def on_eb_depth_change(self, swift_id, direction, old_depth, new_depth):
        """Called when the electronic broker depth changes. Override in subclass."""
        pass

    async def on_new_client_call(self, client_call):
        """Called when a new client call appears. Override in subclass."""
        pass

    async def on_client_call_accepted(self, client_call):
        """Called when a client deal is accepted. Override in subclass."""
        pass

    async def on_client_reject(self, client_call):
        """Called when a client call is rejected. Override in subclass."""
        pass

    async def on_deal_done(self, deal):
        """Called when a deal was done."""
        pass

    async def run(self):
        """Contains the main logic to be executed by the strategy. Override in subclass."""
        pass

    def log_info(self):
        """
        Should be implemented by subclasses to log strategy-specific information.

        Raises:
            NotImplementedError: If the subclass does not implement this method.
        """
        raise NotImplementedError('Strategies must implement the log_info method')
