from .base import Base

# Direction constants
BID, ASK = (0, 1)


class Cover(Base):


    def __init__(self, api, swift_id, options):
        """
        Initializes the Cover strategy with the specified API interface and options.
        """
        super().__init__(api, swift_id,options)

        if 'position_limit' not in options:
            raise Exception(
                f"Cover options dict must contain a key named position_limit with a dictionary of swift:limit pairs.")

        position_limit = options['position_limit']

        if type(position_limit) != int:
            raise Exception(f"{position_limit} is not an integer")
        if position_limit < self.api.min_ticket(swift_id):
            raise Exception(f"{position_limit} is not above min ticket")


        self.position_limit = position_limit
        self.current_amount = self.api.positions[swift_id]['amount']


    async def on_position_change(self, new_position):
        """
        Callback method to handle position changes. Updates the current amount based on the new position.

        :param new_position: The new position data received from the trading system.
        :type new_position: dict
        """
        swift_id = int(new_position['swift_id'])
        if swift_id != int(self.swift_id):
            return

        current_amount = new_position['amount'] / self.api.ticket_unit(swift_id)  # short amount
        self.current_amount = current_amount


    async def run(self):
        """
        Contains the main logic of the strategy. This method checks if the current position exceeds the position
        limit and attempts to adjust the position to align with the defined limit.
        """

        swift_id = self.swift_id

        direction = BID if self.current_amount > 0 else ASK

        remaining_amount = await self.get_cover_amount()

        if remaining_amount <= 0:
            return
        else:
            depth = self.api.get_eb_depth(swift_id)
            ticket = self.api.max_ticket(swift_id) if remaining_amount >= self.api.max_ticket(swift_id) else remaining_amount
            await (self.api.hit_price(swift_id, direction, ticket,
                                         depth[direction][0]['price']))



    async def log_info(self):
        """
        Generates and returns a log message describing the current state of the strategy,
        including the current position amount and the amount to be covered.

        :return: A log message summarizing the current state of the strategy.
        :rtype: str
        """
        return f'COVER   --  Swift: {self.swift_id},  Amount: {self.current_amount}, Limit: {self.position_limit}, To cover: {await self.get_cover_amount()}'

    async def get_cover_amount(self):
        """
        Calculates the amount needed to adjust the current position to meet the position limit.

        :return: The amount that needs to be covered to adhere to the position limit.
        :rtype: int
        """
        return max(abs(self.current_amount) - abs(self.position_limit), 0)
