import pandas as pd
from asgiref.sync import async_to_sync

from .technical_analysis import TechnicalAnalysis

# Direction constants
BEARISH_CROSS, BULLISH_CROSS = (0, 1)


class Macd(TechnicalAnalysis):
    """
    Implements a trading strategy based on the Moving Average Convergence Divergence (MACD) indicator. The MACD is a
    trend-following momentum indicator that shows the relationship between two moving averages of a security's price.
    The MACD is calculated by subtracting the 26-period Exponential Moving Average (EMA) from the 12-period EMA.

    This strategy acts on the signal generated when the MACD line crosses above or below the signal line. A bullish
    crossover (MACD crosses above the signal line) suggests buying, while a bearish crossover (MACD crosses below the
    signal line) suggests selling.

    :param api: The API interface for interacting with the trading system.
    :type api: :class:`traderion.client.TraderionClient`
    :param options: Strategy-specific options including periods for the EMAs used in MACD calculation.
    :type options: dict
    :raises Exception: If required keys (`small_ema_period`, `big_ema_period`, `signal_ema_period`) are missing in `options`.

    BEARISH_CROSS, BULLISH_CROSS = (0, 1)

    """

    def __init__(self, api, swift_id, options):
        """
        Initializes the MACD strategy with the specified API interface and options, including the EMA periods.
        """
        super().__init__(api, swift_id, options)

        if not {'small_ema_period', 'big_ema_period', 'signal_ema_period'} <= set(options):
            raise Exception(f'Macd options should contain the keys: small_ema_period, big_ema_period, '
                            f'signal_ema_period')

        self.small_ema_period = options['small_ema_period']
        self.big_ema_period = options['big_ema_period']
        self.signal_ema_period = options['signal_ema_period']
        options_list = [self.small_ema_period,
                        self.big_ema_period, self.signal_ema_period]

        if not all(isinstance(x, int) and x > 0 for x in options_list):
            raise Exception('Periods must be positive integers')

        self.current_amount = 0
        self.max_ticket = self.api.max_ticket(self.swift_id)
        self.macd = None
        self.signal_line = None
        self.last_cross = None
        async_to_sync( self.compute_macd )()


    async def on_market_price_chart_updated(self, price_curve):
        """
        Responds to changes in the price curve by recalculating the MACD and the signal line.

        :param price_curve: The updated price curve data.
        :type price_curve: list
        """
        self.price_curve = price_curve

        await super().on_market_price_chart_updated(price_curve)
        await self.compute_macd()


    async def compute_macd(self):
        """
        Computes the MACD and the signal line based on the latest price curve. This method updates the
        `macd` and `signal_line` attributes with the latest values.
        """
        small_ema = pd.Series(self.price_curve).ewm(
            span=self.small_ema_period).mean()
        big_ema = pd.Series(self.price_curve).ewm(
            span=self.big_ema_period).mean()
        macd = small_ema - big_ema
        signal_line = macd.ewm(
            span=self.signal_ema_period, adjust=False).mean()
        self.macd = macd.tolist()[-1]
        self.signal_line = signal_line.tolist()[-1]

    async def run(self):
        """
        Executes the strategy's logic. Determines if there's a new MACD cross and updates positions accordingly.
        """
        cross_direction = await self.get_new_cross_direction()

        if cross_direction is not None:
            self.last_cross = cross_direction
            await self.load(cross_direction)


    async def get_new_cross_direction(self):
        """
        Determines the direction of any new MACD cross based on the current values of the MACD line and the signal line.

        :return: The direction of the cross (BEARISH_CROSS or BULLISH_CROSS), or None if no new cross has occurred.
        :rtype: int or None
        """
        cross_direction = None
        if self.last_cross in [BEARISH_CROSS, None] and self.signal_line > self.macd:
            cross_direction = BULLISH_CROSS
        elif self.last_cross in [BULLISH_CROSS, None] and self.signal_line < self.macd:
            cross_direction = BEARISH_CROSS
        return cross_direction

    async def load(self, direction):
        """
        Adjusts the trading position based on the direction of the MACD cross, aiming to reach the target position amount.

        :param direction: The direction to load the position, either BEARISH_CROSS or BULLISH_CROSS.
        :type direction: int
        """
        sgn = (-1, 1)[direction == BULLISH_CROSS]
        remaining_amount = self.target_amount - sgn * self.current_amount
        if remaining_amount < 0:
            return
        else:
            while remaining_amount > 0 and direction == self.last_cross:
                ticket = self.max_ticket if remaining_amount >= self.max_ticket else remaining_amount
                amount = (await self.api.hit_price(self.swift_id, direction, ticket,
                          self.depth[self.swift_id][direction][0]['price']))['data']
                self.current_amount += sgn * amount
                remaining_amount -= amount

    async def log_info(self):
        """
        Generates and returns a log message describing the current state of the strategy, including the current
        position amount, the MACD value, the signal line value, and the last cross direction.

        :return: A log message summarizing the strategy's current state.
        :rtype: str
        """
        last_cross = "Bearish" if self.last_cross == BEARISH_CROSS \
            else "Bullish" if self.last_cross == BULLISH_CROSS else None

        return f'MACD   -- Swift: {self.swift_id}, Amount: {self.current_amount}, Macd: {self.macd}, signal: {self.signal_line}, ' \
               f'Last cross: {last_cross}'
